﻿using System;
using System.Drawing;
using System.IO;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;

namespace NNControl.UPViewer
{
    public class DrawObject
    {
        private UpImageViewer UpViewer;
        private	Rectangle boundingRect;
		private	Point dragPoint;
		private	bool dragging;
        private Bitmap bmp;
        private Bitmap bmpPreview;
        private double zoom = 1.0;
        private bool centerImage;
        private int panelWidth
        {
            get
            {
                return UpViewer.PanelWidth;
            }
        }
        private int panelHeight
        {
            get
            {
                return UpViewer.PanelHeight;
            }
        }
        private int previewPanelWidth
        {
            get
            {
                return UpViewer.PreviewPanelWidth;
            }
        }
        private int previewPanelHeight
        {
            get
            {
                return UpViewer.PreviewPanelHeight;
            }
        }
        private int rotation = 0;
           
        public Rectangle BoundingBox
        {
            get 
            {
                if (centerImage)
                {
                    int w = boundingRect.Width;
                    int h = boundingRect.Height;
                    int x = (UpViewer.PanelWidth - w) / 2;
                    int y = (UpViewer.PanelHeight - h) / 2;
                    if (w >= panelWidth)
                    {
                        if (h >= panelHeight)
                        {
                            return boundingRect;
                        }
                        else
                        {
                            return new Rectangle(boundingRect.X, y, w, h);
                        }
                    }
                    else
                    {
                        if (h >= panelHeight)
                        {
                            return new Rectangle(x, boundingRect.Y, w, h);
                        }
                        else
                        {
                            return new Rectangle(x, y, w, h);
                        }
                    }
                }
                else
                {
                    return boundingRect;
                }
                          
            }
        }

        public void Dispose()
        {
            if (this.Image != null)
            {
                this.Image.Dispose();
            }
        }

        public bool IsDragging
        {
            get { return dragging; }
        }


        public Size OriginalSize
        {
            get
            {
                if (this.Image != null)
                {
                    return this.Image.Size;
                }
                else
                {
                    return Size.Empty;
                }
            }
        }

        public Size CurrentSize
        {
            get { if (boundingRect != null) { return new Size(boundingRect.Width, boundingRect.Height); } else { return Size.Empty; } }
        }
          
        public double Zoom
        {
            get { return zoom; }
        }

        public int Rotation
        {
            get { return rotation; }
            set
            {
                // Making sure that the rotation is only 0, 90, 180 or 270 degrees!
                if (value == 90 || value == 180 || value == 270 || value == 0)
                {
                    rotation = value;
                }
            }
        }

        public int ImageWidth
        {
            get
            {
               
                 return this.Image.Width;
  
            }
        }

        public int ImageHeight
        {
            get
            {
               
                return this.Image.Height;
                
            }
        }

        public Bitmap Image
        {
            get 
            {
                return bmp;
            }
            set
            {
                try
                {
                    if (value != null)
                    {
                       

                        // No memory leaks here!
                        if (this.bmp != null)
                        {
                            this.bmp.Dispose();
                            this.bmp = null;
                        }

                            this.bmp = value;
                      
                        // Initial rotation adjustments
                        if (rotation != 0)
                        {
                            if (rotation == 180)
                            {
                                this.Image.RotateFlip(RotateFlipType.Rotate180FlipNone);
                                boundingRect = new Rectangle(0, 0, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                            }
                            else
                            {
                                int tempWidth = this.Image.Width;
                                int tempHeight = this.Image.Height;
                                if (rotation == 90)
                                {
                                    this.Image.RotateFlip(RotateFlipType.Rotate90FlipNone);
                                }
                                else if (rotation == 270)
                                {
                                    this.Image.RotateFlip(RotateFlipType.Rotate270FlipNone);
                                }

                                // Flip the X and Y values
                                boundingRect = new Rectangle(0, 0, (int)(tempHeight * zoom), (int)(tempWidth * zoom));
                            }
                        }
                        else
                        {
                            this.Image.RotateFlip(RotateFlipType.RotateNoneFlipNone);
                                          
                            boundingRect = new Rectangle(0, 0, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                        }

                        zoom = 1.0;
                        bmpPreview = CreatePreviewImage();
                        FitToScreen();
                    }
                }
                catch (Exception ex)
                {
                    System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
                }
            }
        }
        
        public Image PreviewImage
        {
            get { return bmpPreview; }
        }

        public string ImagePath
        {
            set
            {
                try
                {
                    // No memory leaks here!
                    if (this.bmp != null)
                    {
                        this.bmp.Dispose();
                        this.bmp = null;
                    }

                    Bitmap temp = null;

                    // Make sure it does not crash on incorrect image formats
                    try
                    {
                        //temp = (Bitmap)Bitmap.FromFile(value);
                        temp = new Bitmap(value);
                    }
                    catch
                    {
                        temp = null;
                        System.Windows.Forms.MessageBox.Show("ImageViewer error: Incorrect image format!");
                    }

                    if (temp != null)
                    {
                        
                        this.bmp = temp;
                       

                        // Initial rotation
                        if (rotation != 0)
                        {
                            if (rotation == 180)
                            {
                                this.Image.RotateFlip(RotateFlipType.Rotate180FlipNone);
                                boundingRect = new Rectangle(0, 0, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                            }
                            else
                            {
                                if (rotation == 90)
                                {
                                    this.Image.RotateFlip(RotateFlipType.Rotate90FlipNone);
                                }
                                else if (rotation == 270)
                                {
                                    this.Image.RotateFlip(RotateFlipType.Rotate270FlipNone);
                                }

                                // Flipping X and Y values!
                                boundingRect = new Rectangle(0, 0, (int)(this.ImageHeight * zoom), (int)(this.ImageWidth * zoom));
                            }
                        }
                        else
                        {
                            this.Image.RotateFlip(RotateFlipType.RotateNoneFlipNone);
                            boundingRect = new Rectangle(0, 0, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                        }

                        zoom = 1.0;
                        bmpPreview = CreatePreviewImage();
                        FitToScreen();
                    }
                }
                catch (Exception ex)
                {
                    System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
                }
            }
        }

        public DrawObject(UpImageViewer KpViewer, Bitmap bmp)
        {
            try
            {
                this.UpViewer = KpViewer;

                // Initial dragging to false and an Image.
                dragging = false;
                this.Image = bmp;
                this.Image.RotateFlip(RotateFlipType.RotateNoneFlipNone);
                int w = (int)(this.ImageWidth * zoom);
                int h = (int)(this.ImageHeight * zoom);
                centerImage = true;
                boundingRect = new Rectangle(0, 0, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
            }
            catch(Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        private System.Drawing.Imaging.ImageCodecInfo GetCodec(string type)
        {
            System.Drawing.Imaging.ImageCodecInfo[] info = System.Drawing.Imaging.ImageCodecInfo.GetImageEncoders();

            for (int i = 0; i < info.Length; i++)
            {
                string EnumName = type.ToString();
                if (info[i].FormatDescription.Equals(EnumName))
                {
                    return info[i];
                }
            }
            return null;
        }
        public DrawObject(UpImageViewer KpViewer)
        {
            try
            {
                this.UpViewer = KpViewer;
                // Initial dragging to false and No image.
                dragging = false;
                this.bmp = null;
                centerImage = true;
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void Rotate90()
        {
            try
            {
                if (this.Image != null)
                {
                    int tempWidth = boundingRect.Width;
                    int tempHeight = boundingRect.Height;

                    boundingRect.Width = tempHeight;
                    boundingRect.Height = tempWidth;

                    rotation = (rotation + 90) % 360;
                    this.Image.RotateFlip(RotateFlipType.Rotate90FlipNone);
                    AvoidOutOfScreen();

                    // No memory leaks here!
                    if (this.bmpPreview != null)
                    {
                        this.bmpPreview.Dispose();
                        this.bmpPreview = null;
                    }

                    this.bmpPreview = CreatePreviewImage();
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void Rotate180()
        {
            try
            {
                if (this.Image != null)
                {
                    int tempWidth = boundingRect.Width;
                    int tempHeight = boundingRect.Height;

                    boundingRect.Width = tempHeight;
                    boundingRect.Height = tempWidth;

                    rotation = (rotation + 180) % 360;
                    this.Image.RotateFlip(RotateFlipType.Rotate180FlipNone);
                    AvoidOutOfScreen();

                    // No memory leaks here!
                    if (this.bmpPreview != null)
                    {
                        this.bmpPreview.Dispose();
                        this.bmpPreview = null;
                    }

                    this.bmpPreview = CreatePreviewImage();
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void Rotate270()
        {
            try
            {
                if (this.Image != null)
                {
                    int tempWidth = boundingRect.Width;
                    int tempHeight = boundingRect.Height;

                    boundingRect.Width = tempHeight;
                    boundingRect.Height = tempWidth;

                    rotation = (rotation + 270) % 360;
                    this.Image.RotateFlip(RotateFlipType.Rotate270FlipNone);
                    AvoidOutOfScreen();

                    // No memory leaks here!
                    if (this.bmpPreview != null)
                    {
                        this.bmpPreview.Dispose();
                        this.bmpPreview = null;
                    }

                    this.bmpPreview = CreatePreviewImage();
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        private Bitmap RotateCenter(Bitmap bmpSrc, float theta)
        {
            if(theta == 180.0f)
            {
                Bitmap bmpDest = new Bitmap(bmpSrc.Width, bmpSrc.Height);
                Graphics gDest = Graphics.FromImage(bmpDest);

                gDest.DrawImage(bmpSrc, new Point(0, 0));
                
                bmpDest.RotateFlip(RotateFlipType.Rotate180FlipNone);

                return bmpDest;
            }
            else
            {
                Matrix mRotate = new Matrix();
                mRotate.Translate(bmpSrc.Width / -2, bmpSrc.Height / -2, MatrixOrder.Append);
                mRotate.RotateAt(theta, new Point(0, 0), MatrixOrder.Append);

                GraphicsPath gp = new GraphicsPath();
                // transform image points by rotation matrix
                gp.AddPolygon(new Point[] { new Point(0, 0), new Point(bmpSrc.Width, 0), new Point(0, bmpSrc.Height) });
                gp.Transform(mRotate);
                PointF[] pts = gp.PathPoints;

                // create destination bitmap sized to contain rotated source image
                Rectangle bbox = RotateBoundingBox(bmpSrc, mRotate);
                Bitmap bmpDest = new Bitmap(bbox.Width, bbox.Height);

                Graphics gDest = Graphics.FromImage(bmpDest);
                Matrix mDest = new Matrix();
                mDest.Translate(bmpDest.Width / 2, bmpDest.Height / 2, MatrixOrder.Append);
                gDest.Transform = mDest;
                gDest.DrawImage(bmpSrc, pts);
                gDest.DrawRectangle(Pens.Red, bbox);
                gDest.Dispose();
                gp.Dispose();

                return bmpDest;
            }
        }

        private static Rectangle RotateBoundingBox(Image img, System.Drawing.Drawing2D.Matrix matrix)
        {
            GraphicsUnit gu = new GraphicsUnit();
            Rectangle rImg = Rectangle.Round(img.GetBounds(ref gu));

            // Transform the four points of the image, to get the resized bounding box.
            Point topLeft = new Point(rImg.Left, rImg.Top);
            Point topRight = new Point(rImg.Right, rImg.Top);
            Point bottomRight = new Point(rImg.Right, rImg.Bottom);
            Point bottomLeft = new Point(rImg.Left, rImg.Bottom);
            Point[] points = new Point[] { topLeft, topRight, bottomRight, bottomLeft };
            GraphicsPath gp = new GraphicsPath(points, new byte[] { (byte)PathPointType.Start, (byte)PathPointType.Line, (byte)PathPointType.Line, (byte)PathPointType.Line });
            gp.Transform(matrix);
            return Rectangle.Round(gp.GetBounds());
        }

        private Bitmap CreatePreviewImage()
        {
            // 148 && 117 as initial and default size for the preview panel.
            Rectangle previewRect = new Rectangle(0, 0, previewPanelWidth, previewPanelHeight);

            double x_ratio = (double)previewRect.Width / (double)this.BoundingBox.Width;
            double y_ratio = (double)previewRect.Height / (double)this.BoundingBox.Height;

            if ((this.BoundingBox.Width <= previewRect.Width) && (this.BoundingBox.Height <= previewRect.Height))
            {
                //previewRect.Width = this.BoundingBox.Width;
                //previewRect.Height = this.BoundingBox.Height;
                if (x_ratio > y_ratio)
                {
                    previewRect.Width = (int)Math.Floor((double)this.BoundingBox.Width * (double)y_ratio);

                }
                else
                {
                    previewRect.Height = (int)Math.Floor((double)this.BoundingBox.Height * (double)x_ratio);
                }
            }
            else if ((x_ratio * this.BoundingBox.Height) < previewRect.Height)
            {
                previewRect.Height = Convert.ToInt32(Math.Floor(x_ratio * this.BoundingBox.Height));
                previewRect.Width = previewRect.Width;
            }
            else
            {
                previewRect.Width = Convert.ToInt32(Math.Floor(y_ratio * this.BoundingBox.Width));
                previewRect.Height = previewRect.Height;
            }

            Bitmap bmp = new Bitmap(previewRect.Width, previewRect.Height);

            using (Graphics g = Graphics.FromImage(bmp))
            {
                if (this.Image != null)
                {
                    g.DrawImage(this.Image, previewRect);
                }
            }
            

            return bmp;
        }
        public void CreatePreviewImage(Rectangle rect)
        {
            if (this.Image != null)
            {
                // 148 && 117 as initial and default size for the preview panel.
                Rectangle previewRect = rect;

                double x_ratio = (double)previewRect.Width / (double)this.BoundingBox.Width;
                double y_ratio = (double)previewRect.Height / (double)this.BoundingBox.Height;

                if ((this.BoundingBox.Width <= previewRect.Width) && (this.BoundingBox.Height <= previewRect.Height))
                {
                    //previewRect.Width = this.BoundingBox.Width;
                    //previewRect.Height = this.BoundingBox.Height;
                    if (x_ratio > y_ratio)
                    {
                        previewRect.Width = (int)Math.Floor((double)this.BoundingBox.Width * (double)y_ratio);

                    }
                    else
                    {
                        previewRect.Height = (int)Math.Floor((double)this.BoundingBox.Height * (double)x_ratio);
                    }
                }
                else if ((x_ratio * this.BoundingBox.Height) < previewRect.Height)
                {
                    previewRect.Height = Convert.ToInt32(Math.Floor(x_ratio * this.BoundingBox.Height));
                    previewRect.Width = previewRect.Width;
                }
                else
                {
                    previewRect.Width = Convert.ToInt32(Math.Floor(y_ratio * this.BoundingBox.Width));
                    previewRect.Height = previewRect.Height;
                }

                Bitmap bmp = new Bitmap(previewRect.Width, previewRect.Height);

                using (Graphics g = Graphics.FromImage(bmp))
                {

                    g.DrawImage(this.Image, previewRect);

                }

                // No memory leaks here!
                if (this.bmpPreview != null)
                {
                    this.bmpPreview.Dispose();
                    this.bmpPreview = null;
                }
                this.bmpPreview = bmp;
            }
        }

        public void BoundingBoxResize()
        {
            try
            {
                if (this.Image != null)
                {
                    
                    Point p = PointToOrigin(boundingRect.X, boundingRect.Y, (int)this.ImageWidth, (int)this.ImageHeight);

                    boundingRect = new Rectangle(p.X, p.Y, this.ImageWidth, this.ImageHeight);
                    AvoidOutOfScreen();

                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }
        public void ZoomToSelection(Rectangle selection, Point ptPbFull)
        {
            int x = (selection.X - ptPbFull.X);
            int y = (selection.Y - ptPbFull.Y);
            int width = selection.Width;
            int height = selection.Height;

            // So, where did my selection start on the entire picture?
            int selectedX = (int)((double)(((double)boundingRect.X - ((double)boundingRect.X * 2)) + (double)x) / zoom);
            int selectedY = (int)((double)(((double)boundingRect.Y - ((double)boundingRect.Y * 2)) + (double)y) / zoom);
            int selectedWidth = width;
            int selectedHeight = height;

            // The selection width on the scale of the Original size!
            if (zoom < 1.0 || zoom > 1.0)
            {
                selectedWidth = Convert.ToInt32((double)width / zoom);
                selectedHeight = Convert.ToInt32((double)height / zoom);
            }

            // What is the highest possible zoomrate?
            double zoomX = ((double)panelWidth / (double)selectedWidth);
            double zoomY = ((double)panelHeight / (double)selectedHeight);

            double newZoom = Math.Min(zoomX, zoomY);

            // Avoid Int32 crashes!
            if (newZoom * 100 < Int32.MaxValue && newZoom * 100 > Int32.MinValue)
            {
                SetZoom(newZoom);

                selectedWidth = (int)((double)selectedWidth * newZoom);
                selectedHeight = (int)((double)selectedHeight * newZoom);

                // Center the selected area
                int offsetX = 0;
                int offsetY = 0;
                if (selectedWidth < panelWidth)
                {
                    offsetX = (panelWidth - selectedWidth) / 2;
                }
                if (selectedHeight < panelHeight)
                {
                    offsetY = (panelHeight - selectedHeight) / 2;
                }

                boundingRect.X = (int)((int)((double)selectedX * newZoom) - ((int)((double)selectedX * newZoom) * 2)) + offsetX;
                boundingRect.Y = (int)((int)((double)selectedY * newZoom) - ((int)((double)selectedY * newZoom) * 2)) + offsetY;

                AvoidOutOfScreen();
            }
        }

        public void JumpToOrigin(int x, int y, int width, int height, int pWidth, int pHeight)
        {
            try
            {
                double zoom = (double)boundingRect.Width / (double)width;

                int originX = (int)(x * zoom);
                int originY = (int)(y * zoom);

                originX = originX - (originX * 2);
                originY = originY - (originY * 2);

                boundingRect.X = originX + (pWidth / 2);
                boundingRect.Y = originY + (pHeight / 2);

                AvoidOutOfScreen();
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void JumpToOrigin(int x, int y, int width, int height)
        {
            try
            {
                boundingRect.X = (x - (width / 2)) - ((x - (width / 2)) * 2);
                boundingRect.Y = (y - (height / 2)) - ((y - (height / 2)) * 2);

                AvoidOutOfScreen();
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public Point PointToOrigin(int x, int y, int width, int height)
        {
            try
            {
                double zoomX = (double)width / (double)boundingRect.Width;
                double zoomY = (double)height / (double)boundingRect.Height;

                if (width > panelWidth)
                {
                    int oldX = (boundingRect.X - (boundingRect.X * 2)) + (panelWidth / 2);
                    int oldY = (boundingRect.Y - (boundingRect.Y * 2)) + (panelHeight / 2);

                    int newX = (int)(oldX * zoomX);
                    int newY = (int)(oldY * zoomY);

                    int originX = newX - (panelWidth / 2) - ((newX - (panelWidth / 2)) * 2);
                    int originY = newY - (panelHeight / 2) - ((newY - (panelHeight / 2)) * 2);

                    return new Point(originX, originY);
                }
                else
                {
                    if (height > panelHeight)
                    {
                        int oldY = (boundingRect.Y - (boundingRect.Y * 2)) + (panelHeight / 2);

                        int newY = (int)(oldY * zoomY);

                        int originY = newY - (panelHeight / 2) - ((newY - (panelHeight / 2)) * 2);

                        return new Point(0, originY);
                    }
                    else
                    {
                        return new Point(0, 0);
                    }
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
                return new Point(0, 0);
            }
        }

        public void ZoomIn()
        {
            try
            {
                if (this.Image != null)
                {
                    // Make sure zoom steps are with 25%
                    double index = 0.25 - (zoom % 0.25);
                    
                    if(index != 0)
                    {
                        zoom += index;
                    }
                    else
                    {
                        zoom += 0.25;
                    }

                    Point p = PointToOrigin(boundingRect.X, boundingRect.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));

                    boundingRect = new Rectangle(p.X, p.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                    AvoidOutOfScreen();
                    
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void ZoomOut()
        {
            try
            {
                if (this.Image != null)
                {
                    // Make sure zoom steps are with 25% and higher than 0%
                    if (zoom - 0.25 > 0)
                    {
                        if (((zoom - 0.25) % 0.25) != 0)
                        {
                            zoom -= zoom % 0.25;
                        }
                        else
                        {
                            zoom -= 0.25;
                        }
                    }

                    Point p = PointToOrigin(boundingRect.X, boundingRect.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));

                    boundingRect = new Rectangle(p.X, p.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                    AvoidOutOfScreen();
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void SetPosition(int x, int y)
        {
            boundingRect.X = -x;
            boundingRect.Y = -y;
        }

        public void SetPositionX(int x)
        {
            boundingRect.X = -x;
        }

        public void SetPositionY(int y)
        {
            boundingRect.Y = -y;
        }

        public void SetZoom(double z)
        {
            try
            {
                if (this.Image != null)
                {
                    zoom = z;

                    Point p = PointToOrigin(boundingRect.X, boundingRect.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));

                    boundingRect = new Rectangle(p.X, p.Y, (int)(this.ImageWidth * zoom), (int)(this.ImageHeight * zoom));
                    AvoidOutOfScreen();
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void Scroll(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            try
            {
                if (this.Image != null)
                {
                    if (e.Delta < 0)
                    {
                        ZoomOut();
                    }
                    else
                    {
                        ZoomIn();
                    }
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void FitToScreen()
        {
            try
            {
                if (this.Image != null)
                {
                    double x_ratio = (double)panelWidth / (double)this.ImageWidth;
                    double y_ratio = (double)panelHeight / (double)this.ImageHeight;

                    if ((this.ImageWidth <= panelWidth) && (this.ImageHeight <= panelHeight))
                    {
                        boundingRect.Width = this.ImageWidth;
                        boundingRect.Height = this.ImageHeight;
                    }
                    else if ((x_ratio * this.ImageHeight) < panelHeight)
                    {
                        boundingRect.Height = Convert.ToInt32(Math.Ceiling(x_ratio * this.ImageHeight));
                        boundingRect.Width = panelWidth;
                    }
                    else
                    {
                        boundingRect.Width = Convert.ToInt32(Math.Ceiling(y_ratio * this.ImageWidth));
                        boundingRect.Height = panelHeight;
                    }

                    boundingRect.X = 0;
                    boundingRect.Y = 0;

                    zoom = ((double)boundingRect.Width / (double)this.ImageWidth);
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

        public void AvoidOutOfScreen()
        {
            try
            {
                // Am I lined out to the left?
                if (boundingRect.X >= 0)
                {
                    boundingRect.X = 0;
                }
                else if ((boundingRect.X <= (boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2)))
                {
                    if ((boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2) <= 0)
                    {
                        // I am too far to the left!
                        boundingRect.X = (boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2);
                    }
                    else
                    {
                        // I am too far to the right!
                        boundingRect.X = 0;
                    }
                }

                // Am I lined out to the top?
                if (boundingRect.Y >= 0)
                {
                    boundingRect.Y = 0;
                }
                else if ((boundingRect.Y <= (boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2)))
                {
                    if ((boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2) <= 0)
                    {
                        // I am too far to the top!
                        boundingRect.Y = (boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2);
                    }
                    else
                    {
                        // I am too far to the bottom!
                        boundingRect.Y = 0;
                    }
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
        }

		public void Drag(Point pt)
		{
            try
            {
                if (this.Image != null)
                {
                    if (dragging == true)
                    {
                        // Am I dragging it outside of the panel?
                        if ((pt.X - dragPoint.X > (boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2)) && (pt.X - dragPoint.X < 0))
                        {
                            // No, everything is just fine
                            boundingRect.X = pt.X - dragPoint.X;
                        }
                        else if ((pt.X - dragPoint.X > 0))
                        {
                            // Now don't drag it out of the panel please
                            boundingRect.X = 0;
                        }
                        else if((pt.X - dragPoint.X < (boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2)))
                        {
                            // I am dragging it out of my panel. How many pixels do I have left?
                            if ((boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2) <= 0)
                            {
                                // Make it fit perfectly
                                boundingRect.X = (boundingRect.Width - panelWidth) - ((boundingRect.Width - panelWidth) * 2);
                            }
                        }

                        // Am I dragging it outside of the panel?
                        if (pt.Y - dragPoint.Y > (boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2) && (pt.Y - dragPoint.Y < 0))
                        {
                            // No, everything is just fine
                            boundingRect.Y = pt.Y - dragPoint.Y;
                        }
                        else if ((pt.Y - dragPoint.Y > 0))
                        {
                            // Now don't drag it out of the panel please
                            boundingRect.Y = 0;
                        }
                        else if (pt.Y - dragPoint.Y < (boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2))
                        {
                            // I am dragging it out of my panel. How many pixels do I have left?
                            if ((boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2) <= 0)
                            {
                                // Make it fit perfectly
                                boundingRect.Y = (boundingRect.Height - panelHeight) - ((boundingRect.Height - panelHeight) * 2);
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
		}

		public void BeginDrag(Point pt)
		{
            try
            {
                if (this.Image != null)
                {
                    // Initial drag position

                    dragPoint.X = pt.X - boundingRect.X;
                    dragPoint.Y = pt.Y - boundingRect.Y;
                    dragging = true;
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
		}

		public void EndDrag()
        {
            try
            {
                if (this.Image != null)
                {
                    dragging = false;
                }
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
		}

		public void Draw(Graphics g)
		{
            try
            {
               
                if (this.bmp != null)
                {
                    //draw bitmap to the center
                    int w = boundingRect.Width;
                    int h = boundingRect.Height;
                    int x = (UpViewer.PanelWidth - w) / 2;
                    int y = (UpViewer.PanelHeight - h) / 2;
                    if (centerImage)
                    {
                        if (w >= panelWidth)
                        {
                            if (h >= panelHeight)
                            {
                                g.DrawImage(this.bmp, boundingRect);
                            }
                            else
                            {
                                g.DrawImage(this.bmp, new Rectangle(boundingRect.X, y, w, h));
                            }
                        }
                        else
                        {
                            if (h >= panelHeight)
                            {
                                g.DrawImage(this.bmp, new Rectangle(x, boundingRect.Y, w, h));
                            }
                            else
                            {
                                g.DrawImage(this.bmp, new Rectangle(x, y, w, h));
                            }
                        }
                    }
                    else
                    {
                        g.DrawImage(this.bmp, boundingRect);
                    }
                }
                
            }
            catch (Exception ex)
            {
                System.Windows.Forms.MessageBox.Show("ImageViewer error: " + ex.ToString());
            }
		}
    }
}
